"use client"

import { useState } from "react"
import { TopNav } from "@/components/layout/top-nav"
import { BottomNav } from "@/components/layout/bottom-nav"
import { Avatar, AvatarFallback, AvatarImage } from "@/components/ui/avatar"
import { Button } from "@/components/ui/button"
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs"
import { Heart, MessageCircle, UserPlus, Trophy, Bell, UserCheck } from 'lucide-react'
import { cn } from "@/lib/utils"

interface Notification {
  id: string
  type: "like" | "comment" | "follow" | "friend_request" | "friend_accept" | "mention" | "achievement"
  user?: {
    name: string
    username: string
    avatar: string
  }
  content: string
  timestamp: string
  read: boolean
  postPreview?: string
}

const MOCK_NOTIFICATIONS: Notification[] = [
  {
    id: "0",
    type: "friend_request",
    user: {
      name: "Emily Davis",
      username: "@emily_d",
      avatar: "/placeholder.svg?key=e19k8",
    },
    content: "sent you a Frindle request",
    timestamp: "2m ago",
    read: false,
  },
  {
    id: "1",
    type: "like",
    user: {
      name: "Sarah Johnson",
      username: "@sarah_j",
      avatar: "/placeholder.svg?key=s29k1",
    },
    content: "liked your post",
    timestamp: "5m ago",
    read: false,
    postPreview: "Just launched me first PWA!",
  },
  {
    id: "2",
    type: "comment",
    user: {
      name: "Mike Chen",
      username: "@mike_c",
      avatar: "/placeholder.svg?key=m38d2",
    },
    content: "commented on your post",
    timestamp: "15m ago",
    read: false,
    postPreview: "This is amazing! Great work!",
  },
  {
    id: "3",
    type: "follow",
    user: {
      name: "Alex Rivera",
      username: "@alex_r",
      avatar: "/placeholder.svg?key=a47j3",
    },
    content: "started following you",
    timestamp: "1h ago",
    read: false,
  },
  {
    id: "4",
    type: "achievement",
    content: "You reached 100 followers!",
    timestamp: "2h ago",
    read: true,
  },
  {
    id: "5",
    type: "like",
    user: {
      name: "Emma Wilson",
      username: "@emma_w",
      avatar: "/placeholder.svg?key=e56k4",
    },
    content: "liked your post",
    timestamp: "3h ago",
    read: true,
    postPreview: "Working on some new features",
  },
  {
    id: "6",
    type: "mention",
    user: {
      name: "James Lee",
      username: "@james_l",
      avatar: "/placeholder.svg?key=j65l5",
    },
    content: "mentioned you in a post",
    timestamp: "5h ago",
    read: true,
    postPreview: "Thanks @your_username for the inspiration!",
  },
  {
    id: "7",
    type: "friend_accept",
    user: {
      name: "Kevin Park",
      username: "@kevin_p",
      avatar: "/placeholder.svg?key=k82j9",
    },
    content: "accepted your Frindle request",
    timestamp: "1d ago",
    read: true,
  },
]

export default function AlertsPage() {
  const [activeTab, setActiveTab] = useState("all")
  const [notifications, setNotifications] = useState<Notification[]>(MOCK_NOTIFICATIONS)

  const getNotificationIcon = (type: Notification["type"]) => {
    switch (type) {
      case "like":
        return <Heart className="w-5 h-5 text-destructive fill-destructive" />
      case "comment":
        return <MessageCircle className="w-5 h-5 text-primary" />
      case "follow":
        return <UserPlus className="w-5 h-5 text-primary" />
      case "friend_request":
        return <UserPlus className="w-5 h-5 text-violet-600" />
      case "friend_accept":
        return <UserCheck className="w-5 h-5 text-emerald-600" />
      case "mention":
        return <MessageCircle className="w-5 h-5 text-primary" />
      case "achievement":
        return <Trophy className="w-5 h-5 text-yellow-500" />
    }
  }

  const markAsRead = (id: string) => {
    setNotifications(notifications.map((n) => (n.id === id ? { ...n, read: true } : n)))
  }

  const markAllAsRead = () => {
    setNotifications(notifications.map((n) => ({ ...n, read: true })))
  }

  const filteredNotifications = activeTab === "unread" ? notifications.filter((n) => !n.read) : notifications

  return (
    <div className="min-h-screen pb-20 bg-background">
      <TopNav />
      <main className="max-w-2xl mx-auto">
        <div className="p-4 space-y-4">
          <div className="flex items-center justify-between">
            <div className="flex items-center gap-3">
              <div className="w-10 h-10 rounded-full bg-primary/10 flex items-center justify-center">
                <Bell className="w-6 h-6 text-primary" />
              </div>
              <div>
                <h1 className="text-2xl font-bold text-foreground">Alerts</h1>
                <p className="text-sm text-muted-foreground">{notifications.filter((n) => !n.read).length} unread</p>
              </div>
            </div>

            <Button variant="ghost" size="sm" onClick={markAllAsRead} disabled={!notifications.some((n) => !n.read)}>
              Mark all read
            </Button>
          </div>

          <Tabs value={activeTab} onValueChange={setActiveTab}>
            <TabsList className="w-full grid grid-cols-2">
              <TabsTrigger value="all">All</TabsTrigger>
              <TabsTrigger value="unread">
                Unread{" "}
                {notifications.filter((n) => !n.read).length > 0 && (
                  <span className="ml-2 px-2 py-0.5 bg-destructive text-destructive-foreground text-xs rounded-full">
                    {notifications.filter((n) => !n.read).length}
                  </span>
                )}
              </TabsTrigger>
            </TabsList>

            <TabsContent value={activeTab} className="space-y-2 mt-4">
              {filteredNotifications.length === 0 ? (
                <div className="text-center py-12">
                  <Bell className="w-12 h-12 text-muted-foreground mx-auto mb-4" />
                  <p className="text-muted-foreground">No notifications</p>
                </div>
              ) : (
                filteredNotifications.map((notification) => (
                  <div
                    key={notification.id}
                    className={cn(
                      "bg-card rounded-xl border border-border p-4 cursor-pointer transition-colors hover:bg-secondary",
                      !notification.read && "bg-primary/5 border-primary/20",
                    )}
                    onClick={() => markAsRead(notification.id)}
                  >
                    <div className="flex gap-3">
                      {notification.user ? (
                        <div className="relative">
                          <Avatar className="w-12 h-12">
                            <AvatarImage src={notification.user.avatar || "/placeholder.svg"} />
                            <AvatarFallback>
                              {notification.user.name
                                .split(" ")
                                .map((n) => n[0])
                                .join("")}
                            </AvatarFallback>
                          </Avatar>
                          <div className="absolute -bottom-1 -right-1 w-6 h-6 bg-card rounded-full flex items-center justify-center border-2 border-card">
                            {getNotificationIcon(notification.type)}
                          </div>
                        </div>
                      ) : (
                        <div className="w-12 h-12 rounded-full bg-primary/10 flex items-center justify-center">
                          {getNotificationIcon(notification.type)}
                        </div>
                      )}

                      <div className="flex-1 min-w-0">
                        <div className="flex items-start justify-between gap-2">
                          <p className="text-sm">
                            {notification.user && (
                              <span className="font-semibold text-foreground">{notification.user.name} </span>
                            )}
                            <span
                              className={cn(notification.user ? "text-foreground" : "font-semibold text-foreground")}
                            >
                              {notification.content}
                            </span>
                          </p>
                          {!notification.read && (
                            <div className="w-2 h-2 rounded-full bg-primary flex-shrink-0 mt-1"></div>
                          )}
                        </div>

                        {notification.postPreview && (
                          <p className="text-sm text-muted-foreground mt-1 line-clamp-1">
                            "{notification.postPreview}"
                          </p>
                        )}

                        <p className="text-xs text-muted-foreground mt-1">{notification.timestamp}</p>
                      </div>
                    </div>
                  </div>
                ))
              )}
            </TabsContent>
          </Tabs>
        </div>
      </main>
      <BottomNav />
    </div>
  )
}
