"use client"

import { useEffect, useState } from "react"
import { TopNav } from "@/components/layout/top-nav"
import { BottomNav } from "@/components/layout/bottom-nav"
import { ChatSidebar } from "@/components/layout/chat-sidebar"
import { InlineCreatePost } from "@/components/posts/inline-create-post"
import { ThreadView } from "@/components/posts/thread-view"
import { PostCard } from "@/components/posts/post-card"
import { registerServiceWorker } from "@/lib/pwa-register"
import type { Post, Reply } from "@/types/post"

const MOCK_POSTS: Post[] = [
  {
    id: "1",
    author: {
      id: "1",
      username: "digital_nomad",
      displayName: "Digital Nomad",
      avatar: "/diverse-user-avatars.png",
    },
    content:
      "Just launched me first PWA! The feeling of seeing your code come to life is unmatched. The feeling of seeing your code come...",
    type: "thought",
    timestamp: "2m ago",
    likes: 12,
    comments: 3,
    isLiked: false,
    privacy: "public",
    replies: [],
  },
  {
    id: "2",
    author: {
      id: "1",
      username: "digital_nomad",
      displayName: "Digital Nomad",
      avatar: "/diverse-user-avatars.png",
    },
    content:
      "Just launched me first PWA! The feeling of seeing your code come to life is unmatched. The feeling of seeing your code come...",
    type: "thought",
    timestamp: "2m ago",
    likes: 12,
    comments: 3,
    isLiked: false,
    privacy: "public",
    replies: [],
  },
  {
    id: "3",
    author: {
      id: "1",
      username: "digital_nomad",
      displayName: "Digital Nomad",
      avatar: "/diverse-user-avatars.png",
    },
    content:
      "Just launched me first PWA! The feeling of seeing your code come to life is unmatched. The feeling of seeing your code come...",
    type: "thought",
    timestamp: "2m ago",
    likes: 12,
    comments: 3,
    isLiked: false,
    privacy: "public",
    replies: [],
  },
]

export default function Home() {
  const [posts, setPosts] = useState<Post[]>(MOCK_POSTS)
  const [selectedPost, setSelectedPost] = useState<Post | null>(null)
  const [isThreadViewOpen, setIsThreadViewOpen] = useState(false)

  useEffect(() => {
    registerServiceWorker()
  }, [])

  const handleNewPost = (post: Post) => {
    setPosts([post, ...posts])
  }

  const handleLike = (postId: string) => {
    setPosts(
      posts.map((post) =>
        post.id === postId
          ? { ...post, isLiked: !post.isLiked, likes: post.isLiked ? post.likes - 1 : post.likes + 1 }
          : post,
      ),
    )
  }

  const handleReply = (postId: string, reply: Reply) => {
    setPosts(
      posts.map((post) =>
        post.id === postId
          ? { ...post, replies: [...(post.replies || []), reply], comments: (post.replies?.length || 0) + 1 }
          : post,
      ),
    )
  }

  const handleOpenThread = (post: Post) => {
    setSelectedPost(post)
    setIsThreadViewOpen(true)
  }

  return (
    <div className="flex min-h-screen bg-background">
      <div className="fixed inset-0 dark:bg-gradient-to-br dark:from-[#0a0e13] dark:via-[#151a26] dark:to-[#1a1f2e] -z-10" />

      <div className="flex-1 min-h-screen pb-20">
        <TopNav />
        <main className="max-w-4xl mx-auto lg:max-w-[calc(100vw-420px)] lg:ml-8 lg:mr-8">
          <div className="p-4 space-y-4">
            <InlineCreatePost onPost={handleNewPost} />

            <div className="space-y-2">
              <h2 className="text-lg font-semibold text-accent">Latest Posts</h2>
              {posts.map((post) => (
                <PostCard key={post.id} post={post} onLike={handleLike} onClick={() => handleOpenThread(post)} />
              ))}
            </div>
          </div>
        </main>
        <BottomNav />

        <ThreadView
          post={selectedPost}
          open={isThreadViewOpen}
          onOpenChange={setIsThreadViewOpen}
          onLike={handleLike}
          onReply={handleReply}
        />
      </div>
      <ChatSidebar />
    </div>
  )
}
