"use client"

import { useState } from "react"
import { TopNav } from "@/components/layout/top-nav"
import { BottomNav } from "@/components/layout/bottom-nav"
import { Avatar, AvatarFallback, AvatarImage } from "@/components/ui/avatar"
import { Button } from "@/components/ui/button"
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs"
import { Alert, AlertDescription } from "@/components/ui/alert"
import { Badge } from "@/components/ui/badge"
import { cn } from "@/lib/utils"

// SVG Icons
const TrophyIcon = () => (
  <svg className="w-6 h-6" fill="none" viewBox="0 0 24 24" stroke="currentColor">
    <path
      strokeLinecap="round"
      strokeLinejoin="round"
      strokeWidth={2}
      d="M9 12l2 2 4-4M7.835 4.697a3.42 3.42 0 001.946-.806 3.42 3.42 0 014.438 0 3.42 3.42 0 001.946.806 3.42 3.42 0 013.138 3.138 3.42 3.42 0 00.806 1.946 3.42 3.42 0 010 4.438 3.42 3.42 0 00-.806 1.946 3.42 3.42 0 01-3.138 3.138 3.42 3.42 0 00-1.946.806 3.42 3.42 0 01-4.438 0 3.42 3.42 0 00-1.946-.806 3.42 3.42 0 01-3.138-3.138 3.42 3.42 0 00-.806-1.946 3.42 3.42 0 010-4.438 3.42 3.42 0 00.806-1.946 3.42 3.42 0 013.138-3.138z"
    />
  </svg>
)

const MedalIcon = () => (
  <svg className="w-6 h-6" fill="none" viewBox="0 0 24 24" stroke="currentColor">
    <path
      strokeLinecap="round"
      strokeLinejoin="round"
      strokeWidth={2}
      d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"
    />
  </svg>
)

const StarIcon = ({ filled = false }: { filled?: boolean }) => (
  <svg className="w-4 h-4" fill={filled ? "currentColor" : "none"} viewBox="0 0 24 24" stroke="currentColor">
    <path
      strokeLinecap="round"
      strokeLinejoin="round"
      strokeWidth={2}
      d="M11.049 2.927c.3-.921 1.603-.921 1.902 0l1.519 4.674a1 1 0 00.95.69h4.915c.969 0 1.371 1.24.588 1.81l-3.976 2.888a1 1 0 00-.363 1.118l1.518 4.674c.3.922-.755 1.688-1.538 1.118l-3.976-2.888a1 1 0 00-1.176 0l-3.976 2.888c-.783.57-1.838-.197-1.538-1.118l1.518-4.674a1 1 0 00-.363-1.118l-3.976-2.888c-.784-.57-.38-1.81.588-1.81h4.914a1 1 0 00.951-.69l1.519-4.674z"
    />
  </svg>
)

const TrendingIcon = ({ direction }: { direction: "up" | "down" }) => (
  <svg
    className={cn("w-3 h-3", direction === "down" && "rotate-180")}
    fill="none"
    viewBox="0 0 24 24"
    stroke="currentColor"
  >
    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M13 7h8m0 0v8m0-8l-8 8-4-4-6 6" />
  </svg>
)

const AlertIcon = () => (
  <svg className="w-4 h-4" fill="none" viewBox="0 0 24 24" stroke="currentColor">
    <path
      strokeLinecap="round"
      strokeLinejoin="round"
      strokeWidth={2}
      d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-3L13.732 4c-.77-1.333-2.694-1.333-3.464 0L3.34 16c-.77 1.333.192 3 1.732 3z"
    />
  </svg>
)

const HeartIcon = () => (
  <svg className="w-4 h-4" fill="none" viewBox="0 0 24 24" stroke="currentColor">
    <path
      strokeLinecap="round"
      strokeLinejoin="round"
      strokeWidth={2}
      d="M4.318 6.318a4.5 4.5 0 000 6.364L12 20.364l7.682-7.682a4.5 4.5 0 00-6.364-6.364L12 7.636l-1.318-1.318a4.5 4.5 0 00-6.364 0z"
    />
  </svg>
)

interface LeaderboardUser {
  id: string
  rank: number
  name: string
  username: string
  avatar: string
  starsRetained: number
  starsLost: number
  totalStars: number
  positivityScore: number
  posts: number
  change: number
  status: "positive" | "neutral" | "warning" | "negative"
  warningMessage?: string
}

const MOCK_LEADERS: LeaderboardUser[] = [
  {
    id: "1",
    rank: 1,
    name: "Sarah Johnson",
    username: "@sarah_j",
    avatar: "/placeholder.svg?height=100&width=100",
    starsRetained: 5,
    starsLost: 0,
    totalStars: 5,
    positivityScore: 98,
    posts: 324,
    change: 2,
    status: "positive",
  },
  {
    id: "2",
    rank: 2,
    name: "Mike Chen",
    username: "@mike_c",
    avatar: "/placeholder.svg?height=100&width=100",
    starsRetained: 5,
    starsLost: 0,
    totalStars: 5,
    positivityScore: 96,
    posts: 298,
    change: 1,
    status: "positive",
  },
  {
    id: "3",
    rank: 3,
    name: "Alex Rivera",
    username: "@alex_r",
    avatar: "/placeholder.svg?height=100&width=100",
    starsRetained: 4,
    starsLost: 1,
    totalStars: 5,
    positivityScore: 92,
    posts: 267,
    change: 1,
    status: "positive",
  },
  {
    id: "4",
    rank: 4,
    name: "Emma Wilson",
    username: "@emma_w",
    avatar: "/placeholder.svg?height=100&width=100",
    starsRetained: 4,
    starsLost: 1,
    totalStars: 5,
    positivityScore: 89,
    posts: 245,
    change: 0,
    status: "neutral",
  },
  {
    id: "5",
    rank: 5,
    name: "James Lee",
    username: "@james_l",
    avatar: "/placeholder.svg?height=100&width=100",
    starsRetained: 3,
    starsLost: 2,
    totalStars: 5,
    positivityScore: 75,
    posts: 231,
    change: -1,
    status: "neutral",
  },
  {
    id: "6",
    rank: 6,
    name: "Sophie Anderson",
    username: "@sophie_a",
    avatar: "/placeholder.svg?height=100&width=100",
    starsRetained: 2,
    starsLost: 3,
    totalStars: 5,
    positivityScore: 58,
    posts: 218,
    change: -2,
    status: "warning",
    warningMessage: "Let's bring positivity back. Ready to change that tone?",
  },
  {
    id: "7",
    rank: 7,
    name: "David Kim",
    username: "@david_k",
    avatar: "/placeholder.svg?height=100&width=100",
    starsRetained: 1,
    starsLost: 4,
    totalStars: 5,
    positivityScore: 42,
    posts: 189,
    change: -3,
    status: "negative",
    warningMessage: "Your tone matters. Let's uplift, not tear down.",
  },
]

export default function LeadersPage() {
  const [activeTab, setActiveTab] = useState("all")
  const [leaders] = useState<LeaderboardUser[]>(MOCK_LEADERS)

  const [currentUser] = useState({
    rank: 42,
    starsUsed: 4,
    totalStars: 5,
    negativePostsToday: 4,
    showWarning: true,
  })

  const getRankIcon = (rank: number) => {
    switch (rank) {
      case 1:
        return (
          <div className="text-yellow-500">
            <TrophyIcon />
          </div>
        )
      case 2:
        return (
          <div className="text-gray-400">
            <MedalIcon />
          </div>
        )
      case 3:
        return (
          <div className="text-amber-700">
            <MedalIcon />
          </div>
        )
      default:
        return <span className="text-lg font-bold text-muted-foreground">{rank}</span>
    }
  }

  const renderStars = (retained: number, lost: number) => {
    return (
      <div className="flex items-center gap-0.5">
        {Array.from({ length: 5 }).map((_, i) => {
          if (i < retained) {
            return (
              <div key={i} className="text-yellow-500">
                <StarIcon filled />
              </div>
            )
          } else {
            return (
              <div key={i} className="text-gray-300 dark:text-gray-600">
                <StarIcon />
              </div>
            )
          }
        })}
      </div>
    )
  }

  const getStatusBadge = (status: string) => {
    switch (status) {
      case "positive":
        return (
          <Badge className="bg-green-500/10 text-green-600 dark:text-green-400 border-green-500/20">
            <div className="mr-1">
              <HeartIcon />
            </div>
            Positive
          </Badge>
        )
      case "warning":
        return (
          <Badge className="bg-orange-500/10 text-orange-600 dark:text-orange-400 border-orange-500/20">
            <div className="mr-1">
              <AlertIcon />
            </div>
            Warning
          </Badge>
        )
      case "negative":
        return (
          <Badge className="bg-red-500/10 text-red-600 dark:text-red-400 border-red-500/20">
            <div className="mr-1">
              <AlertIcon />
            </div>
            Needs Improvement
          </Badge>
        )
      default:
        return null
    }
  }

  return (
    <div className="min-h-screen pb-20 bg-background">
      <TopNav />
      <main className="max-w-2xl mx-auto">
        <div className="p-4 space-y-4">
          <div className="flex items-center gap-3">
            <div className="w-10 h-10 rounded-full bg-primary/10 flex items-center justify-center text-primary">
              <TrophyIcon />
            </div>
            <div>
              <h1 className="text-2xl font-bold text-foreground">Leaderboard</h1>
              <p className="text-sm text-muted-foreground">Ranked by positivity and star retention</p>
            </div>
          </div>

          {currentUser.showWarning && (
            <Alert className="bg-orange-500/10 border-orange-500/30">
              <div className="text-orange-600 dark:text-orange-400">
                <AlertIcon />
              </div>
              <AlertDescription className="text-orange-800 dark:text-orange-200">
                You've used {currentUser.negativePostsToday} of {currentUser.totalStars} stars on negative posts today.
                Choose your next words wisely.
              </AlertDescription>
            </Alert>
          )}

          <Tabs value={activeTab} onValueChange={setActiveTab}>
            <TabsList className="w-full grid grid-cols-3">
              <TabsTrigger value="all">All Time</TabsTrigger>
              <TabsTrigger value="week">This Week</TabsTrigger>
              <TabsTrigger value="month">This Month</TabsTrigger>
            </TabsList>

            <TabsContent value={activeTab} className="space-y-2 mt-4">
              {leaders.map((user, index) => (
                <div key={user.id} className="space-y-2">
                  <div
                    className={cn(
                      "bg-card rounded-xl border border-border p-4",
                      index < 3 && "bg-primary/5 border-primary/20",
                      user.status === "negative" && "bg-red-500/5 border-red-500/20",
                      user.status === "warning" && "bg-orange-500/5 border-orange-500/20",
                    )}
                  >
                    <div className="flex items-center gap-4">
                      <div className="w-12 flex items-center justify-center">{getRankIcon(user.rank)}</div>

                      <Avatar className="w-12 h-12">
                        <AvatarImage src={user.avatar || "/placeholder.svg"} />
                        <AvatarFallback>
                          {user.name
                            .split(" ")
                            .map((n) => n[0])
                            .join("")}
                        </AvatarFallback>
                      </Avatar>

                      <div className="flex-1 min-w-0 space-y-1">
                        <div className="flex items-center gap-2">
                          <h3 className="font-semibold text-foreground">{user.name}</h3>
                          {getStatusBadge(user.status)}
                        </div>
                        <p className="text-sm text-muted-foreground">{user.username}</p>

                        <div className="flex items-center gap-2">
                          {renderStars(user.starsRetained, user.starsLost)}
                          <span className="text-xs text-muted-foreground">
                            {user.starsRetained}/{user.totalStars} stars
                          </span>
                        </div>
                      </div>

                      <div className="text-right">
                        <div className="font-bold text-lg text-foreground">{user.positivityScore}%</div>
                        <div className="flex items-center justify-end gap-1">
                          {user.change > 0 ? (
                            <>
                              <div className="text-green-500">
                                <TrendingIcon direction="up" />
                              </div>
                              <span className="text-xs text-green-500">+{user.change}</span>
                            </>
                          ) : user.change < 0 ? (
                            <>
                              <div className="text-red-500">
                                <TrendingIcon direction="down" />
                              </div>
                              <span className="text-xs text-red-500">{user.change}</span>
                            </>
                          ) : (
                            <span className="text-xs text-muted-foreground">-</span>
                          )}
                        </div>
                      </div>
                    </div>

                    {user.warningMessage && (
                      <div className="mt-3 pt-3 border-t border-border">
                        <p className="text-sm text-muted-foreground italic">{user.warningMessage}</p>
                      </div>
                    )}
                  </div>
                </div>
              ))}
            </TabsContent>
          </Tabs>

          <div className="bg-primary/10 rounded-xl border border-primary/20 p-4">
            <div className="flex items-center justify-between mb-2">
              <span className="text-sm font-medium text-foreground">Your Rank</span>
              <span className="text-sm text-muted-foreground">#{currentUser.rank}</span>
            </div>
            <div className="flex items-center gap-3">
              <Avatar className="w-10 h-10">
                <AvatarImage src="/placeholder.svg?height=100&width=100" />
                <AvatarFallback>You</AvatarFallback>
              </Avatar>
              <div className="flex-1 space-y-1">
                <div className="flex items-center gap-2">
                  {renderStars(currentUser.totalStars - currentUser.starsUsed, currentUser.starsUsed)}
                  <span className="text-xs text-muted-foreground">
                    {currentUser.totalStars - currentUser.starsUsed}/{currentUser.totalStars} stars remaining
                  </span>
                </div>
                <span className="text-xs text-muted-foreground">
                  {currentUser.negativePostsToday} negative posts today
                </span>
              </div>
              <Button size="sm">View Stats</Button>
            </div>
          </div>

          <div className="bg-card rounded-xl border border-border p-4 space-y-2">
            <h3 className="font-semibold text-foreground flex items-center gap-2">
              <div className="text-yellow-500">
                <StarIcon filled />
              </div>
              How the Star System Works
            </h3>
            <div className="space-y-2 text-sm text-muted-foreground">
              <p>• Everyone starts with 5 stars daily</p>
              <p>• Positive posts help you retain stars and climb the leaderboard</p>
              <p>• Negative posts cost you stars and lower your ranking</p>
              <p>• Consistency and positivity build your social standing over time</p>
            </div>
          </div>
        </div>
      </main>
      <BottomNav />
    </div>
  )
}
