"use client"

import { useState } from "react"
import { TopNav } from "@/components/layout/top-nav"
import { BottomNav } from "@/components/layout/bottom-nav"
import { Avatar, AvatarFallback, AvatarImage } from "@/components/ui/avatar"
import { Input } from "@/components/ui/input"
import { Search } from "lucide-react"
import Link from "next/link"
import { cn } from "@/lib/utils"

interface Conversation {
  id: string
  user: {
    id: string
    name: string
    username: string
    avatar: string
  }
  lastMessage: string
  timestamp: string
  unread: boolean
}

const MOCK_CONVERSATIONS: Conversation[] = [
  {
    id: "1",
    user: {
      id: "1",
      name: "Marvin McKinney",
      username: "marvin_m",
      avatar: "/placeholder.svg?key=m2k2d",
    },
    lastMessage: "You: Thank you",
    timestamp: "7:36 pm",
    unread: false,
  },
  {
    id: "2",
    user: {
      id: "2",
      name: "Marvin McKinney",
      username: "marvin_m",
      avatar: "/placeholder.svg?key=0skc9",
    },
    lastMessage: "You: Thank you",
    timestamp: "7:36 pm",
    unread: false,
  },
  {
    id: "3",
    user: {
      id: "3",
      name: "Marvin McKinney",
      username: "marvin_m",
      avatar: "/placeholder.svg?key=2j0k1",
    },
    lastMessage: "You: Thank you",
    timestamp: "7:36 pm",
    unread: false,
  },
  {
    id: "4",
    user: {
      id: "4",
      name: "Marvin McKinney",
      username: "marvin_m",
      avatar: "/placeholder.svg?key=3mn1k",
    },
    lastMessage: "You: Thank you",
    timestamp: "7:36 pm",
    unread: false,
  },
]

export default function MessagesPage() {
  const [searchQuery, setSearchQuery] = useState("")
  const [conversations] = useState<Conversation[]>(MOCK_CONVERSATIONS)

  const filteredConversations = conversations.filter(
    (conv) =>
      conv.user.name.toLowerCase().includes(searchQuery.toLowerCase()) ||
      conv.user.username.toLowerCase().includes(searchQuery.toLowerCase()),
  )

  return (
    <div className="min-h-screen pb-20 bg-background">
      <TopNav />
      <main className="max-w-2xl mx-auto">
        <div className="p-4 space-y-4">
          <div className="flex items-center justify-between">
            <h1 className="text-2xl font-bold text-primary">Messages</h1>
          </div>

          <div className="relative">
            <Search className="absolute left-3 top-1/2 -translate-y-1/2 w-5 h-5 text-muted-foreground" />
            <Input
              placeholder="Search messages..."
              value={searchQuery}
              onChange={(e) => setSearchQuery(e.target.value)}
              className="pl-10 h-12"
            />
          </div>

          <div className="space-y-1">
            {filteredConversations.map((conversation) => (
              <Link
                key={conversation.id}
                href={`/messages/${conversation.id}`}
                className={cn(
                  "flex items-center gap-3 p-3 rounded-lg hover:bg-secondary transition-colors",
                  conversation.unread && "bg-secondary/50",
                )}
              >
                <Avatar className="w-12 h-12">
                  <AvatarImage src={conversation.user.avatar || "/placeholder.svg"} />
                  <AvatarFallback>
                    {conversation.user.name
                      .split(" ")
                      .map((n) => n[0])
                      .join("")}
                  </AvatarFallback>
                </Avatar>

                <div className="flex-1 min-w-0">
                  <div className="flex items-center justify-between">
                    <span
                      className={cn(
                        "font-semibold text-sm",
                        conversation.unread ? "text-foreground" : "text-foreground",
                      )}
                    >
                      {conversation.user.name}
                    </span>
                    <span className="text-xs text-muted-foreground">{conversation.timestamp}</span>
                  </div>
                  <p
                    className={cn(
                      "text-sm truncate",
                      conversation.unread ? "text-foreground font-medium" : "text-muted-foreground",
                    )}
                  >
                    {conversation.lastMessage}
                  </p>
                </div>
              </Link>
            ))}
          </div>
        </div>
      </main>
      <BottomNav />
    </div>
  )
}
