"use client"

import { Button } from "@/components/ui/button"
import { UserPlus, UserCheck, UserX, Clock, Users } from 'lucide-react'
import type { FrindleStatus } from "@/types/frindle"
import { cn } from "@/lib/utils"

interface FrindleButtonProps {
  userId: string
  status: FrindleStatus
  onSendRequest?: (userId: string) => void
  onAcceptRequest?: (userId: string) => void
  onRejectRequest?: (userId: string) => void
  onCancelRequest?: (userId: string) => void
  onRemoveConnection?: (userId: string) => void
  size?: "sm" | "default" | "lg"
  variant?: "default" | "outline" | "ghost"
}

export function FrindleButton({
  userId,
  status,
  onSendRequest,
  onAcceptRequest,
  onRejectRequest,
  onCancelRequest,
  onRemoveConnection,
  size = "sm",
  variant = "default",
}: FrindleButtonProps) {
  const handleClick = (e: React.MouseEvent) => {
    e.stopPropagation()
    
    switch (status) {
      case "none":
        onSendRequest?.(userId)
        break
      case "pending_sent":
        onCancelRequest?.(userId)
        break
      case "connected":
        onRemoveConnection?.(userId)
        break
    }
  }

  const getButtonContent = () => {
    switch (status) {
      case "none":
        return {
          icon: <UserPlus className="w-4 h-4" />,
          text: "Add Frindle",
          variant: "default" as const,
          className: "bg-gradient-to-r from-violet-600 to-indigo-600 hover:from-violet-700 hover:to-indigo-700",
        }
      case "pending_sent":
        return {
          icon: <Clock className="w-4 h-4" />,
          text: "Pending",
          variant: "outline" as const,
          className: "border-yellow-500 text-yellow-600 hover:bg-yellow-50",
        }
      case "pending_received":
        return {
          icon: <UserCheck className="w-4 h-4" />,
          text: "Accept",
          variant: "default" as const,
          className: "bg-gradient-to-r from-emerald-600 to-teal-600 hover:from-emerald-700 hover:to-teal-700",
        }
      case "connected":
        return {
          icon: <Users className="w-4 h-4" />,
          text: "Frindles",
          variant: "outline" as const,
          className: "border-green-500 text-green-600 hover:bg-green-50",
        }
      case "rejected":
        return {
          icon: <UserX className="w-4 h-4" />,
          text: "Rejected",
          variant: "ghost" as const,
          className: "text-muted-foreground cursor-not-allowed",
        }
      default:
        return {
          icon: <UserPlus className="w-4 h-4" />,
          text: "Add Frindle",
          variant: "default" as const,
          className: "bg-gradient-to-r from-violet-600 to-indigo-600 hover:from-violet-700 hover:to-indigo-700",
        }
    }
  }

  const content = getButtonContent()

  // Special rendering for pending_received status (show both Accept and Reject)
  if (status === "pending_received") {
    return (
      <div className="flex gap-2">
        <Button
          size={size}
          onClick={(e) => {
            e.stopPropagation()
            onAcceptRequest?.(userId)
          }}
          className="gap-2 bg-gradient-to-r from-emerald-600 to-teal-600 hover:from-emerald-700 hover:to-teal-700"
        >
          <UserCheck className="w-4 h-4" />
          Accept
        </Button>
        <Button
          size={size}
          variant="outline"
          onClick={(e) => {
            e.stopPropagation()
            onRejectRequest?.(userId)
          }}
          className="gap-2 border-red-500 text-red-600 hover:bg-red-50"
        >
          <UserX className="w-4 h-4" />
          Decline
        </Button>
      </div>
    )
  }

  return (
    <Button
      size={size}
      variant={content.variant}
      onClick={handleClick}
      disabled={status === "rejected"}
      className={cn("gap-2", content.className)}
    >
      {content.icon}
      {content.text}
    </Button>
  )
}
