"use client"

import { Avatar, AvatarFallback, AvatarImage } from "@/components/ui/avatar"
import { Button } from "@/components/ui/button"
import { Heart, MessageCircle, Music } from 'lucide-react'
import type { Post } from "@/types/post"
import { cn } from "@/lib/utils"
import Link from "next/link"

interface PostCardProps {
  post: Post
  onLike: (postId: string) => void
  onClick?: () => void
}

export function PostCard({ post, onLike, onClick }: PostCardProps) {
  return (
    <div className="bg-card rounded-xl border border-border p-4 space-y-4">
      <div className="flex gap-3">
        <Link href={`/profile/${post.author.id}`}>
          <Avatar className="w-10 h-10 cursor-pointer hover:opacity-80 transition-opacity">
            <AvatarImage src={post.author.avatar || "/placeholder.svg"} />
            <AvatarFallback>{post.author.displayName[0]}</AvatarFallback>
          </Avatar>
        </Link>

        <div className="flex-1 min-w-0">
          <div className="flex items-center gap-2">
            <Link 
              href={`/profile/${post.author.id}`}
              className="font-semibold text-sm text-foreground hover:underline cursor-pointer"
            >
              {post.author.displayName}
            </Link>
            <span className="text-xs text-muted-foreground">{post.timestamp}</span>
          </div>
          <Link 
            href={`/profile/${post.author.id}`}
            className="text-sm text-muted-foreground hover:underline cursor-pointer"
          >
            @{post.author.username}
          </Link>
        </div>
      </div>

      <p className="text-base text-foreground leading-relaxed">{post.content}</p>

      {post.imageUrl && (
        <div className="rounded-lg overflow-hidden">
          <img src={post.imageUrl || "/placeholder.svg"} alt="Post image" className="w-full h-auto object-cover" />
        </div>
      )}

      {post.musicUrl && (
        <div className="bg-secondary rounded-lg p-4 flex items-center gap-4">
          <div className="w-12 h-12 bg-primary/20 rounded flex items-center justify-center">
            <Music className="w-6 h-6 text-primary" />
          </div>
          <div className="flex-1">
            <p className="font-semibold text-sm">{post.musicTitle}</p>
            <p className="text-xs text-muted-foreground">{post.musicArtist}</p>
          </div>
        </div>
      )}

      <div className="flex items-center gap-6 pt-2">
        <Button
          variant="ghost"
          size="sm"
          onClick={() => onLike(post.id)}
          className={cn("gap-2 px-2", post.isLiked && "text-destructive hover:text-destructive")}
        >
          <Heart className={cn("w-5 h-5", post.isLiked && "fill-current")} />
          <span className="text-sm font-medium">{post.likes}</span>
        </Button>

        <Button variant="ghost" size="sm" className="gap-2 px-2" onClick={onClick}>
          <MessageCircle className="w-5 h-5" />
          <span className="text-sm font-medium">{post.replies?.length || post.comments}</span>
        </Button>
      </div>
    </div>
  )
}
