"use client"

import { useState } from "react"
import { Button } from "@/components/ui/button"
import { Textarea } from "@/components/ui/textarea"
import { Avatar, AvatarFallback, AvatarImage } from "@/components/ui/avatar"
import { Plus, ImageIcon, Music } from "lucide-react"
import type { Post } from "@/types/post"

interface PostComposerProps {
  onPost: (post: Post) => void
}

export function PostComposer({ onPost }: PostComposerProps) {
  const [content, setContent] = useState("")
  const [postType, setPostType] = useState<"thought" | "photo" | "music">("thought")
  const [isExpanded, setIsExpanded] = useState(false)

  const handleSubmit = () => {
    if (!content.trim()) return

    const newPost: Post = {
      id: Date.now().toString(),
      author: {
        id: "current-user",
        username: "you",
        displayName: "You",
        avatar: "/diverse-user-avatars.png",
      },
      content,
      type: postType,
      timestamp: "Just now",
      likes: 0,
      comments: 0,
      isLiked: false,
    }

    onPost(newPost)
    setContent("")
    setIsExpanded(false)
    setPostType("thought")
  }

  return (
    <div className="bg-card rounded-xl border border-border p-4 space-y-4">
      <div className="flex gap-3">
        <Avatar className="w-10 h-10">
          <AvatarImage src="/diverse-user-avatars.png" />
          <AvatarFallback>You</AvatarFallback>
        </Avatar>

        <div className="flex-1">
          <Textarea
            placeholder="What's interesting..."
            value={content}
            onChange={(e) => setContent(e.target.value)}
            onFocus={() => setIsExpanded(true)}
            className="min-h-[60px] resize-none border-0 focus-visible:ring-0 p-0 text-base bg-transparent"
          />
        </div>
      </div>

      {isExpanded && (
        <div className="space-y-4">
          <div className="flex gap-2">
            <Button
              variant={postType === "thought" ? "default" : "outline"}
              size="sm"
              onClick={() => setPostType("thought")}
              className="gap-2"
            >
              <Plus className="w-4 h-4" />
              Thought
            </Button>
            <Button
              variant={postType === "photo" ? "default" : "outline"}
              size="sm"
              onClick={() => setPostType("photo")}
              className="gap-2"
            >
              <ImageIcon className="w-4 h-4" />
              Photo
            </Button>
            <Button
              variant={postType === "music" ? "default" : "outline"}
              size="sm"
              onClick={() => setPostType("music")}
              className="gap-2"
            >
              <Music className="w-4 h-4" />
              Music
            </Button>
          </div>

          <div className="flex justify-end">
            <Button
              onClick={handleSubmit}
              disabled={!content.trim()}
              className="bg-destructive hover:bg-destructive/90 text-white px-8"
            >
              Post
            </Button>
          </div>
        </div>
      )}
    </div>
  )
}
